/***********************************************************************************************//**
 * @file		main.cpp
 * @brief		The main file of an example code for CAN Gateway with hdj2534.so (v1.0.226)
 * @date		2014-03-06
 * @copyright	Hatteland Display AS
 * @mainpage	Hatteland Display CAN Gateway example for Linux
 **************************************************************************************************/

//==================================================================================================
// Includes
//==================================================================================================
#include <iostream>
#include <cstring>
#include <unistd.h>
#include <pthread.h>
#include <hdj2534.h>
#include "cangw.hpp"

//==================================================================================================
// Namespaces
//==================================================================================================


//==================================================================================================
// Defines
//==================================================================================================


//==================================================================================================
// Typedefs
//==================================================================================================


//==================================================================================================
// Declarations of local functions
//==================================================================================================
static void rcvChannel1(const J2534::PASSTHRU_MSG * rxPassThruMsg, unsigned long rxMsgNum);
static void rcvChannel2(const J2534::PASSTHRU_MSG * rxPassThruMsg, unsigned long rxMsgNum);

//==================================================================================================
// Variables
//==================================================================================================
/// Number of messages to send
static const unsigned int MESSAGES_TO_SEND = 50;
/// Period for periodic messages
static const unsigned int PERIODIC_MSG_INTERVAL = 50;
/// Marking for messages sent over channel 1
static const char CHANNEL_1_MARK = '1';
/// Marking for messages sent over channel 2
static const char CHANNEL_2_MARK = '2';
/// Marking for periodic messages
static const char PERIODIC_MSG_MARK = 'P';
/// Mutex used by callback functions during accessing the screen
static pthread_mutex_t screenMutex = PTHREAD_MUTEX_INITIALIZER;

//==================================================================================================
//==================================================================================================

/***********************************************************************************************//**
 * @brief The main function
 **************************************************************************************************/
int main()
{
	using namespace std;

	if (canGw::CAN_GW_OK == canGw::init(canGw::BAUD_125K, canGw::BAUD_125K, &rcvChannel1,
			&rcvChannel2))
	{
		J2534::PASSTHRU_MSG txPassThruMsg;
		J2534::J2534_TxFlags txFlags;
		unsigned int id;

		memset(&txPassThruMsg, 0, sizeof(J2534::PASSTHRU_MSG));
		txFlags.bits.Can29BitId = 1;
		txPassThruMsg.TxFlags = txFlags.value;
		txPassThruMsg.ProtocolID = J2534::CAN;
		id = 0x0000ABCD;
		txPassThruMsg.Data[0] = static_cast<unsigned char>(id >> 24);
		txPassThruMsg.Data[1] = static_cast<unsigned char>(id >> 16);
		txPassThruMsg.Data[2] = static_cast<unsigned char>(id >> 8);
		txPassThruMsg.Data[3] = static_cast<unsigned char>(id >> 0);
		txPassThruMsg.Data[4] = CHANNEL_1_MARK; // source of a message - channel 1
		txPassThruMsg.Data[5] = PERIODIC_MSG_MARK; // marking periodic message
		txPassThruMsg.DataSize = 6;
		if (canGw::CAN_GW_OK != canGw::startPeriodic(canGw::CHANNEL_1, &txPassThruMsg,
				PERIODIC_MSG_INTERVAL))
		{
			cout << "CAN_GW periodic msg start ERROR! " << endl;
		}
		id = 0x0000BCDE;
		txPassThruMsg.Data[0] = static_cast<unsigned char>(id >> 24);
		txPassThruMsg.Data[1] = static_cast<unsigned char>(id >> 16);
		txPassThruMsg.Data[2] = static_cast<unsigned char>(id >> 8);
		txPassThruMsg.Data[3] = static_cast<unsigned char>(id >> 0);
		txPassThruMsg.Data[4] = CHANNEL_2_MARK; // source of a message - channel 2
		if (canGw::CAN_GW_OK != canGw::startPeriodic(canGw::CHANNEL_2, &txPassThruMsg,
				PERIODIC_MSG_INTERVAL))
		{
			cout << "CAN_GW periodic msg start ERROR! " << endl;
		}

		id = 0x00000055;
		for (unsigned int i = 0; i < MESSAGES_TO_SEND; i++)
		{
			txPassThruMsg.Data[0] = static_cast<unsigned char>(id >> 24);
			txPassThruMsg.Data[1] = static_cast<unsigned char>(id >> 16);
			txPassThruMsg.Data[2] = static_cast<unsigned char>(id >> 8);
			txPassThruMsg.Data[3] = static_cast<unsigned char>(id >> 0);
			txPassThruMsg.Data[4] = CHANNEL_1_MARK; // source of a message - channel 1
			txPassThruMsg.Data[5] = '0';
			txPassThruMsg.Data[6] = '0';
			txPassThruMsg.Data[7] = '0';
			txPassThruMsg.Data[8] = '0';
			txPassThruMsg.Data[9] = '0';
			txPassThruMsg.Data[10] = (i / 10) + '0'; // a message number on each channel
			txPassThruMsg.Data[11] = (i % 10) + '0'; // a message number on each channel
			txPassThruMsg.DataSize = 12;
			if (canGw::CAN_GW_OK != canGw::send(canGw::CHANNEL_1, &txPassThruMsg))
			{
				cout << "CAN_GW send ERROR! " << endl;
				break;
			}
			txPassThruMsg.Data[4] = CHANNEL_2_MARK; // source of a message - channel 2
			if (canGw::CAN_GW_OK != canGw::send(canGw::CHANNEL_2, &txPassThruMsg))
			{
				cout << "CAN_GW send ERROR! " << endl;
				break;
			}
			// 5 ms delay for main loop
			static const unsigned int LOOP_DELAY = 5 * 1000;
			usleep(LOOP_DELAY);

			id++;
		}
		if (canGw::CAN_GW_OK != canGw::uninit())
		{
			cout << "CAN_GW uninit ERROR!" << endl;
		}
	}
	else
	{
		cout << "CAN_GW init ERROR!" << endl;
		cout << "Has application got superuser privileges?" << endl;
		cout << "Is CAN Gateway connected?" << endl;
	}
	return 0;
}

//==================================================================================================
//==================================================================================================

/***********************************************************************************************//**
 * @brief Callback function for receive on channel 1
 * @param[in] rxPassThruMsg Pointer to buffer with received messages
 * @param[in] rxMsgNum Number of received messages
 **************************************************************************************************/
static void rcvChannel1(const J2534::PASSTHRU_MSG * rxPassThruMsg, unsigned long rxMsgNum)
{
	using namespace std;

	for (unsigned int i = 0; i < rxMsgNum; i++)
	{
		pthread_mutex_lock(&screenMutex);
		cout << "CHANNEL_1_cb | msg_src:ch" << rxPassThruMsg[i].Data[4];
		cout << " | id:" << static_cast<unsigned int>(rxPassThruMsg[i].Data[0]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[1]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[2]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[3]);
		if (PERIODIC_MSG_MARK == rxPassThruMsg[i].Data[5])
		{
			cout << " | periodic";
		}
		cout << endl;
		pthread_mutex_unlock(&screenMutex);
	}
}
/***********************************************************************************************//**
 * @brief Callback function for receive on channel 2
 * @param[in] rxPassThruMsg Pointer to buffer with received messages
 * @param[in] rxMsgNum Number of received messages
 **************************************************************************************************/
static void rcvChannel2(const J2534::PASSTHRU_MSG * rxPassThruMsg, unsigned long rxMsgNum)
{
	using namespace std;

	for (unsigned int i = 0; i < rxMsgNum; i++)
	{
		pthread_mutex_lock(&screenMutex);
		cout << "CHANNEL_2_cb | msg_src:ch" << rxPassThruMsg[i].Data[4];
		cout << " | id:" << static_cast<unsigned int>(rxPassThruMsg[i].Data[0]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[1]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[2]);
		cout << " " << static_cast<unsigned int>(rxPassThruMsg[i].Data[3]);
		if (PERIODIC_MSG_MARK == rxPassThruMsg[i].Data[5])
		{
			cout << " | periodic";
		}
		cout << endl;
		pthread_mutex_unlock(&screenMutex);
	}
}
/**************************************************************************************************/
